<?php
/**
 * Plugin Name: Hazard Test Plugin
 * Description: A WordPress plugin for interactive hazard perception tests.
 * Version: 1.0
 * Author: Your Name
 */

// Prevent direct access
if (!defined('ABSPATH')) exit;

// Plugin activation
function htp_create_video_table() {
    global $wpdb;

    $table_name = $wpdb->prefix . 'htp_videos';

    // Check if the table already exists
    if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") !== $table_name) {
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE $table_name (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            title VARCHAR(255) NOT NULL,
            video_url VARCHAR(255) NOT NULL,
            danger_points TEXT NOT NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP
        ) $charset_collate;";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);
    }
}

// Hook into `init`
add_action('init', 'htp_create_video_table');
function htp_create_video_table_once() {
    global $wpdb;

    $table_name = $wpdb->prefix . 'htp_videos';

    // Check if the table creation has been done
    if (get_option('htp_video_table_created')) {
        return;
    }

    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE $table_name (
        id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        video_url VARCHAR(255) NOT NULL,
        danger_points TEXT NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    ) $charset_collate;";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta($sql);

    // Mark as created
    update_option('htp_video_table_created', true);
}

// Hook into `init`
add_action('init', 'htp_create_video_table_once');

function htp_add_admin_menu() {
    add_menu_page(
        'Hazard Test', 
        'Hazard Test', 
        'manage_options', 
        'hazard-test', 
        'htp_admin_page', 
        'dashicons-media-video'
    );
}
add_action('admin_menu', 'htp_add_admin_menu');

function htp_admin_page() {
    global $wpdb;

    // Table name for videos
    $table_name = $wpdb->prefix . 'htp_videos';

    // Handle video upload
    if (isset($_POST['htp_save_video'])) {
        htp_save_video();
    }

    // Fetch all videos
    $videos = $wpdb->get_results("SELECT * FROM $table_name ORDER BY created_at DESC");

    // Styles for the admin page
    ?>
    <style>
        .htp-container {
            font-family: Arial, sans-serif;
            margin: 20px 0;
        }
        .htp-form-container {
            background: #f9f9f9;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 8px;
            margin-bottom: 20px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.1);
        }
        .htp-form-container h2 {
            margin-top: 0;
        }
        .htp-form-container input[type="text"],
        .htp-form-container input[type="file"] {
            width: calc(100% - 20px);
            padding: 10px;
            margin: 10px 0;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        .htp-form-container input[type="submit"] {
            background: #0073aa;
            color: #fff;
            padding: 10px 20px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
        }
        .htp-form-container input[type="submit"]:hover {
            background: #005c8a;
        }
        .htp-video-list {
            margin-top: 20px;
        }
        .htp-video-card {
            background: #fff;
            border: 1px solid #ddd;
            border-radius: 8px;
            margin-bottom: 15px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.1);
            overflow: hidden;
            display: flex;
        }
        .htp-video-thumbnail {
            flex: 1;
            background: #000;
        }
        .htp-video-thumbnail video {
            width: 100%;
            height: 100%;
        }
        .htp-video-details {
            flex: 2;
            padding: 10px 15px;
            display: flex;
            flex-direction: column;
            justify-content: space-between;
        }
        .htp-video-title {
            font-weight: bold;
            font-size: 16px;
        }
        .htp-video-shortcode {
            background: #f4f4f4;
            padding: 5px 10px;
            border-radius: 4px;
            font-family: monospace;
            font-size: 14px;
            margin-top: 5px;
        }
        .htp-video-actions a {
            color: #d63638;
            text-decoration: none;
            font-weight: bold;
            margin-top: 10px;
            display: inline-block;
        }
        .htp-video-actions a:hover {
            text-decoration: underline;
        }
    </style>
<div class="htp-container">
    <div class="htp-form-container">
        <h2>Add New Video</h2>
        <form method="post" enctype="multipart/form-data" action="">
            <label for="htp_video_title">Video Title:</label>
            <input type="text" name="htp_video_title" placeholder="Enter video title" required>

            <label for="htp_video_url">Video File:</label>
            <input type="file" name="htp_video_url" accept="video/*" required>

            <label for="htp_danger_points">Danger Points (comma-separated in seconds):</label>
            <input type="text" name="htp_danger_points" placeholder="e.g., 5,10,20" required>

            <input type="submit" name="htp_save_video" value="Save Video">
        </form>
    </div>

    <div class="htp-video-list">
        <h2>Uploaded Videos</h2>
        <?php if ($videos): ?>
            <?php foreach ($videos as $video): ?>
                <div class="htp-video-card">
                    <div class="htp-video-thumbnail">
                        <video controls>
                            <source src="<?php echo esc_url($video->video_url); ?>" type="video/mp4">
                            Your browser does not support the video tag.
                        </video>
                    </div>
                    <div class="htp-video-details">
                        <div class="htp-video-title"><?php echo esc_html($video->title); ?></div>
                        <div class="htp-video-shortcode">
                            Shortcode: <code>[htp_video id="<?php echo esc_attr($video->id); ?>"]</code>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php else: ?>
            <p>No videos uploaded yet.</p>
        <?php endif; ?>
    </div>
</div>
<?php
}

function htp_save_video() {
    global $wpdb;

    $table_name = $wpdb->prefix . 'htp_videos';

    // Check if the form data is set
    if (!isset($_POST['htp_video_title'], $_FILES['htp_video_url'], $_POST['htp_danger_points'])) {
        return;
    }

    // Validate the video upload
    if ($_FILES['htp_video_url']['error'] !== UPLOAD_ERR_OK) {
        echo '<p style="color: red;">Error uploading video. Please try again.</p>';
        return;
    }

    // Handle video upload
    $upload = wp_upload_bits(
        $_FILES['htp_video_url']['name'],
        null,
        file_get_contents($_FILES['htp_video_url']['tmp_name'])
    );

    if ($upload['error']) {
        echo '<p style="color: red;">Error saving video file: ' . esc_html($upload['error']) . '</p>';
        return;
    }

    // Prepare data for database
    $title = sanitize_text_field($_POST['htp_video_title']);
    $video_url = esc_url($upload['url']);
    $danger_points = sanitize_text_field($_POST['htp_danger_points']);
    $danger_points_array = array_map('intval', explode(',', $danger_points));

    // Insert data into the database
    $wpdb->insert(
        $table_name,
        [
            'title' => $title,
            'video_url' => $video_url,
            'danger_points' => json_encode($danger_points_array),
            'created_at' => current_time('mysql'),
        ]
    );

    if ($wpdb->insert_id) {
        echo '<p style="color: green;">Video saved successfully!</p>';
    } else {
        echo '<p style="color: red;">Error saving video to the database.</p>';
    }
}

function htp_video_shortcode($atts) {
    global $wpdb;

    $atts = shortcode_atts(['id' => 0], $atts, 'htp_video');
    $video = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM {$wpdb->prefix}htp_videos WHERE id = %d", $atts['id']
    ));

    if (!$video) return '<p>Video not found.</p>';

    $danger_points = json_decode($video->danger_points, true);

    ob_start();
    ?>
    <div id="htp-video-<?php echo esc_attr($video->id); ?>" class="htp-video">
        <video id="video-<?php echo esc_attr($video->id); ?>" width="640" height="360" controls>
            <source src="<?php echo esc_url($video->video_url); ?>" type="video/mp4">
            Your browser does not support the video tag.
        </video>
        <div class="htp-overlay">Click to Start</div>
        <div class="htp-countdown" style="display: none;">3</div>
    </div>
    <script>
        (function($) {
            const video = document.getElementById('video-<?php echo esc_js($video->id); ?>');
            const overlay = $('.htp-overlay');
            const countdown = $('.htp-countdown');

            overlay.click(function() {
                overlay.hide();
                let counter = 3;
                countdown.show().text(counter);
                const interval = setInterval(() => {
                    counter--;
                    countdown.text(counter);
                    if (counter <= 0) {
                        countdown.hide();
                        video.play();
                        clearInterval(interval);
                    }
                }, 1000);
            });

            const dangerPoints = <?php echo json_encode($danger_points); ?>;
            video.addEventListener('timeupdate', () => {
                const currentTime = Math.floor(video.currentTime);
                if (dangerPoints.includes(currentTime)) {
                    console.log('Danger point detected at:', currentTime);
                }
            });
        })(jQuery);
    </script>
    <?php
    return ob_get_clean();
}
add_shortcode('htp_video', 'htp_video_shortcode');

function htp_render_video_shortcode($atts) {  
    global $wpdb;  

    $atts = shortcode_atts(['id' => 0], $atts, 'htp_video');  
    $video_id = intval($atts['id']);  

    if (!$video_id) {  
        return '<p>Invalid video ID.</p>';  
    }  

    // Fetch video details  
    $table_name = $wpdb->prefix . 'htp_videos';  
    $video = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $video_id));  

    if (!$video) {  
        return '<p>Video not found.</p>';  
    }  

    // Parse danger points  
    $danger_points = json_decode($video->danger_points, true);  

    // Render the video with interactive overlay  
    ob_start();  
    ?>  
    <div class="htp-video-container">  
        <div id="htp-video-overlay">  
            <p>You shouldn't skip the video</p>  
            <button id="htp-start-button">Start</button>  
            <div id="countdown"></div> <!-- Countdown display -->  
        </div>  
        <video id="htp-video" width="640" height="360" muted>  
            <source src="<?php echo esc_url($video->video_url); ?>" type="video/mp4">  
            Your browser does not support the video tag.  
        </video>  
        <div id="htp-results"></div>  
    </div>  
    <script>  
        document.addEventListener("DOMContentLoaded", function () {  
            const video = document.getElementById("htp-video");  
            const startButton = document.getElementById("htp-start-button");  
            const overlay = document.getElementById("htp-video-overlay");  
            const countdownDisplay = document.getElementById("countdown");  
            const results = document.getElementById("htp-results");  

            let dangerPoints = <?php echo json_encode($danger_points); ?>;  
            let userClicks = [];  
            const tolerance = 1; // 1-second tolerance for clicks  
            let clickedTimes = new Set();  // Use a Set to track unique clicks  

            // Countdown and start video on button click  
            startButton.addEventListener("click", function () {  
                overlay.style.display = "none"; // Hide overlay with instructions  
                countdownDisplay.style.display = "block"; // Show the countdown  

                let countdownValue = 3; // Start at 3  
                const countdownInterval = setInterval(function () {  
                    countdownDisplay.textContent = countdownValue;  
                    countdownDisplay.classList.remove('go'); // Remove GO! class for countdown  
                    if (countdownValue === 0) {  
                        countdownDisplay.textContent = "GO!";  
                        countdownDisplay.classList.add('go'); // Add GO! class for styling  
                        clearInterval(countdownInterval); // Stop the countdown  
                        video.play(); // Start the video  
                    }  
                    countdownValue--;  
                }, 1000); // Update every second  
            });  

            // Record user clicks  
            video.addEventListener("click", function () {  
                const currentTime = Math.floor(video.currentTime); // Round to nearest second  
                
                // If the click time is not already recorded, add it  
                if (!clickedTimes.has(currentTime)) {  
                    clickedTimes.add(currentTime);  
                    userClicks.push(currentTime);  
                }  
            });  

            // Process results when video ends  
            video.addEventListener("ended", function () {  
                let correctClicks = 0;  

                // Compare user clicks to danger points  
                userClicks.forEach(click => {  
                    // Check if the click is within the tolerance of any danger point  
                    if (dangerPoints.some(dp => Math.abs(dp - click) <= tolerance)) {  
                        correctClicks++;  
                    }  
                });  

                // Calculate score based on correct clicks (out of total danger points)  
                const totalDangerPoints = dangerPoints.length;  
                const score = Math.min((correctClicks / totalDangerPoints) * 100, 100); // Limit to 100%  

                // Display results  
                const result = score >= 60 ? 'Pass' : 'Fail';  
                results.innerHTML = `  
                    <p>Your score: ${Math.round(score)}%</p>  
                    <p>Result: ${result}</p>  
                `;  
            });  

            // Hide video controls  
            video.controls = false;  
        });  
    </script>  
    <style>  
        .htp-video-container {  
            position: relative;  
            width: 640px;  
            margin: 0 auto;  
        }  
        #htp-video-overlay {  
            position: absolute;  
            top: 0;  
            left: 0;  
            width: 100%;  
            height: 100%;  
            background: rgba(0, 0, 0, 0.7);  
            display: flex;  
            justify-content: center;  
            align-items: center;  
            flex-direction: column;  
            color: white;  
            z-index: 10;  
            text-align: center;  
        }  
        #htp-video-overlay p {  
            font-size: 18px;  
            margin-bottom: 20px;  
        }  
        #htp-video-overlay button {  
            padding: 10px 20px;  
            font-size: 18px;  
            background-color: #28a745;  
            color: white;  
            border: none;  
            border-radius: 5px;  
            cursor: pointer;  
        }  
        #htp-video-overlay button:hover {  
            background-color: #218838;  
        }  
        #countdown {  
            background: rgba(0, 0, 0, 0.7);  
            font-size: 100px; /* Increased font size */  
            font-weight: bold;  
            color: white;  
            display: none;  
            transition: opacity 0.5s ease; /* Smooth transition */  
        }  
        #countdown.go {  
            animation: flash 1s infinite; /* Flashing effect for GO! */  
        }  
        @keyframes flash {  
            0%, 100% {  
                opacity: 1;  
            }  
            50% {  
                opacity: 0.5;  
            }  
        }  
        #htp-results {  
            margin-top: 10px;  
            text-align: center;  
            font-size: 18px;  
            font-weight: bold;  
        }  
    </style>  
    <?php  
    return ob_get_clean();  
}  

add_shortcode('htp_video', 'htp_render_video_shortcode');
